/**************************************************************************
*           Copyright (c) 2001, Cisco Systems, All Rights Reserved
***************************************************************************
*
*  File:    linuxcniapi.c
*  Date:    22/03/01
*
***************************************************************************
* This module implements a translation layer between the CNI API and the
* Linux Interceptor driver.
***************************************************************************/
#include <linux/config.h>
#include <linux/version.h>
#include <linux/netdevice.h>
#include <linux/if.h>
#include <linux/if_arp.h>
#include <linux/if_ether.h>
#include <linux/skbuff.h>
#include <linux/etherdevice.h>
#include <linux/ip.h>
#include <linux/vmalloc.h>
#include "linux_os.h"

#include "Cniapi.h"
#include "unixcniapi.h"
#include "linuxcniapi.h"
/******************************** Globals *********************************/

CNI_CHARACTERISTICS CNICallbackTable; /* This stores the Plugin's function pointers */
PCHAR pcDeviceName;             /* Ignore. Only our pluggin so we don't care about it */


void *
os_alloc(size_t size)
{
    return kmalloc(size, GFP_ATOMIC);
}

void
os_free(void *p, size_t size)
{
    kfree(p);
}

/*************************************************************************
*
* CniGetFrameType
*
* Description:
* This function retrieves the media type for the packets sent and received
* via this medium.
*
*	
* Argument:
*
*   IN CNIBINDING Binding - A binding handle.
*   OUT PNICMEDIUM pMedium - The media type for this binding.
*                CniMedium802_3	Ethernet
*                CniMedium802_5	Token Ring
*                CniMediumFddi	Fiber Distributed Data Interface
*                CniMediumWan	Various point to point and WAN interfaces
*                CniMediumLocalTalk	LoacalTalk network (MAC)
*                CniMediumDix	Ethernet - drivers use the DIX header format.
*                CniMediumArcnetRaw	ARCNET network
*                CniMediumArcnet878_2	ARCNET 878.2 network
*                CniMediumAtm	ATM network
*                CniMediumWirelessWan	Various wireless WAN networks
*                CniMediumIrda	IRDA, infrared network
*
*  NOTE - Only CniMediumUNKNOWN, CniMedium802_3 and CniMediumWan will be
*         returned at present as only the second 2 are accepted. Other
*         is for errors (shouldn't happen).
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - The Frame type was returned.
*                CNI_E_BAD_BINDING - The Binding is not a CNIBINDING.
*                CNI_E_BAD_PARAMETER - pMedium was NULL.
*
*************************************************************************/
CNISTATUS
CniGetFrameType(IN CNIBINDING Binding, OUT PNICMEDIUM pMedium)
{

    // We will support 802_3 for now.
    *pMedium = CniMedium802_3;


    return CNI_SUCCESS;
}

 /*************************************************************************
*
* CniGetMacAddress
*
* Description:
* This function retrieves the MAC address for the binding specified.
* The character array represented by *ppMacAddress must not be modified,
* or released.
*
*
* Argument:	
*
*   IN CNIBINDING Binding - A binding handle
*   OUT PCHAR *ppMacAddress - A pointer to a character array that
*                contains the MAC address.
*   OUT ULONG pulMacAddressSize - The Fragment length was changed.
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - Returned the MAC address.
*                CNI_E_BAD_BINDING - The Binding is not a CNIBINDING.
*                CNI_E_BAD_PARAMETER - pMedium was NULL.
*
*************************************************************************/
extern uint8 ppp_fake_lcl_addr[ETH_ALEN];
CNISTATUS
CniGetMacAddress(IN CNIBINDING Binding,
                 OUT PCHAR * ppMacAddress, OUT ULONG * pulMacAddressSize)
{
    //  this code will be different in 2.4.x kernel
    PBINDING pBinding;

    pBinding = (PBINDING) Binding;

    if (!pBinding || !pBinding->pDevice)
    {
        return CNI_E_BAD_BINDING;
    }

    if (!ppMacAddress || !pulMacAddressSize)
    {
        return CNI_E_BAD_PARAMETER;
    }

    *ppMacAddress = pBinding->pDevice->dev_addr;

    *pulMacAddressSize = ETH_ALEN;

    return CNI_SUCCESS;
}

/*************************************************************************
*
* CniGetMacName
*
* Description:
* This function retrieves the name of the network device associated with a
* given binding.  It works differently depending on the operating system.
*
* On NT systems, the MAC name will be the full device name of the NIC card,
* as seen by the IPCONFIG.EXE application.  The name string on NT includes
* the full device driver name for the NIC driver, which has the form:
* "DosDevices\<driver name>".  The registry settings for the device can be
*  found in the registry at:
* "HKEY_LOCAL_MACNIE\System\CurrentControlSet\Services\<driver name>"
*
* On Windows 95/98 systems, the name string is an enumerated string
* corresponding to the NIC driver.  Settings for the NIC driver are found in
* the registry at:
* "HKEY_LOCAL_MACHINE\System\CurrentControlSet\Services\Class\Net\<enumerated string>"
*
*
*
* Argument:
*	
*   IN CNIBINDING Binding - A binding handle that will be used to determine
*                           the associated network adapter
*   OUT PCHAR *macName - a pointer to the character string giving the name
*                        of the name associated with Binding
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - The packet was sent by the NIC.
*                CNI_E_BAD_PARAMETER - One of the parameters was invalid
*
*************************************************************************/
CNISTATUS
CniGetMacName(IN CNIBINDING Binding, OUT PCHAR * pszMacName)
{
    PBINDING pBinding;

    pBinding = (PBINDING) Binding;

    if (!pBinding || !pszMacName || !pBinding->pDevice)
    {
        return CNI_E_BAD_PARAMETER;
    }


    *pszMacName = pBinding->pDevice->name;

    return CNI_SUCCESS;
}

/*************************************************************************
*
* CniGetMTUSize
*
* Description:
* This function retrieves the MTU for the specified binding.
*
*	
* Argument:
*
*   IN CNIBINDING Binding - A binding handle
*   OUT PULONG pulMtuSize - The size, in bytes, of the MTU for
*                this binding.
*
* Returns:
*
*   CNISTATUS  - CNI_SUCCESS - The MTU was retrieved.
*                CNI_E_BAD_BINDING - The Binding is not a CNIBINDING.
*                CNI_E_BAD_PARAMETER - pulMtuSize was NULL.
*
*************************************************************************/
CNISTATUS
CniGetMTUSize(IN CNIBINDING Binding, OUT PULONG pulMtuSize)
{
    PBINDING pBinding;

    if (!Binding || !pulMtuSize)
        return CNI_E_BAD_PARAMETER;

    pBinding = (PBINDING) Binding;

    /* return the original MTU */
    *pulMtuSize = (ULONG) pBinding->original_mtu;

    return CNI_SUCCESS;
}


/*************************************************************************
* 
* CniInjectReceive
* 
* Description:
* This function will inject a Packet into the receive data stream as though
* they were received by and send to a NIC/protocol pair defined by Binding.
* The ReceiveContext should be set to provide packet-specific information 
* to CniTransferData() and CniReceiveComplete().  This plugin will not see
* the injected packet at its CniReceive() interface, it will be sent 
* directly to the protocol.
* If the size of Packet is less than ulSize, and the receiving protocol is
* interested in receiving the rest of the packet the plugin's
* CniTransferData() entry point will be called to copy the remaining packet
* data into a Packet belonging to the protocol.
* 
* Argument:
* 
*   IN CNIBINDING Binding - A binding handle that will be used to
*                receive this packet.
*   IN PVOID ReceiveContext -	Data to be passed to the 
*                TransferData entry point.
*   IN CNIFRAGMENT MacHeader - This is the header that will be 
*                copied to the packet before it is received by the protocol.
*                It should correspond to the Binding.
*   IN CNIPACKET Packet - The packet that will be received, minus 
*                the MAC Header.
*   IN ULONG ulSize - The size of the complete packet.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The packet was sent
*                CNI_PENDING - The packet was not sent yet, but will be.
*                CNI_W_BAD_FORMAT - The packet data was formatted incorrectly.
*                CNI_E_NO_ENTRY_POINT - The plugin did not have a 
*                CniTransferData() entry point.
* 
*************************************************************************/
CNISTATUS
CniInjectReceive(IN CNIBINDING Binding,
                 IN PVOID ReceiveContext,
                 IN CNIFRAGMENT MacHeader, IN CNIPACKET Packet, IN ULONG ulSize)
{
    CNISTATUS rc = CNI_SUCCESS;
    LPPACKETDESCRIPTOR lpPacketDescriptor;
    PBINDING pBinding;
    LPFRAGMENTBUFFER lpMacFragment;
    struct sk_buff *skb = NULL;
    unsigned char *pIP = NULL, *pMac = NULL;


    /* we need to build the actual sk_buff from the packet structure */
    pBinding = (PBINDING) Binding;
    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;
    lpMacFragment = (LPFRAGMENTBUFFER) MacHeader;

    skb = dev_alloc_skb(lpPacketDescriptor->uiPacketSize
                        + lpMacFragment->uiFragmentDataSize);
    if (!skb)
    {
        rc = CNI_W_OUT_OF_DESCRIPTORS;
        goto exit_gracefully;
    }
    /* move the data into the packet */
    do_gettimeofday(&skb->stamp);

    pIP = skb_put(skb, lpPacketDescriptor->uiPacketSize);

    CniGetPacketData(Packet, 0, lpPacketDescriptor->uiPacketSize, pIP);

    skb->dev = pBinding->pDevice;

    /* put back the mac header */
    switch (pBinding->real_hh_len)
    {
    case ETH_HLEN:
        pMac = skb_push(skb, lpMacFragment->uiFragmentDataSize);
        memcpy(pMac, lpMacFragment->lpFragmentData,
               lpMacFragment->uiFragmentDataSize);

        skb->protocol = eth_type_trans(skb, skb->dev);
        break;
    case IPPP_MAX_HEADER:
    case 0:
        pMac = pIP;
        skb->protocol = htons(ETH_P_IP);
        break;
    default:
        break;
    }
    skb->dev = pBinding->pDevice;



    skb->ip_summed = CHECKSUM_UNNECESSARY;

    skb->nh.iph = (struct iphdr *) skb->data;
    skb->mac.raw = pMac;

    pBinding->recv_stat.called = TRUE;

    pBinding->recv_stat.rc =
        pBinding->InjectReceive(skb, skb->dev, pBinding->pPT);
  exit_gracefully:
    CNICallbackTable.ReceiveComplete(Binding, ReceiveContext, Packet);
    return rc;
}


/*************************************************************************
* 
* CniInjectSend
* 
* Description:
* This call will inject a packet into the send data stream, the packet will
* appear to be originating from and sent via a protocol/NIC pair defined by
* Binding.  This plugin will not see the packet in it CniSend() entry point,
* but will receive notification that the packet has been sent via it's 
* CniSendComplete() entry point.  The SendContext will be used to send 
* packet-specific data to CniSendComplete so that this packet may be 
* identified as belonging to this plugin.
*
* 
* 
* Argument:
*	
*   IN DNBINDING Binding - A binding handle that will be used
*                 to receive this packet.
*   IN PVOID SendContext - Data to be passed to the SendComplete
*                 entry point.
*   IN DNFRAGMENT MacHeader - This is the header that will be 
*                copied to the packet before it is sent over the NIC.  It 
*                should correspond to the Binding.
*   IN DNPACKET Packet - The packet that will be sent, minus the
*                 MAC header.
* 
* Returns:
* 
*   CNISTATUS  - CNI_SUCCESS - The packet was sent by the NIC.
*                CNI_PENDING - The packet will be sent by theNIC.
*                CNI_W_BAD_FORMAT - The packet data was not formatted 
*                correctly for the NIC.
* 
*************************************************************************/
CNISTATUS
CniInjectSend(IN CNIBINDING Binding,
              IN PVOID SendContext,
              IN CNIFRAGMENT MacHeader, IN CNIPACKET Packet)
{
    CNISTATUS rc = CNI_SUCCESS;
    LPPACKETDESCRIPTOR lpPacketDescriptor;
    PBINDING pBinding;
    LPFRAGMENTBUFFER lpMacFragment;
    struct sk_buff *skb;
    unsigned char *pIP = NULL, *pMac = NULL;

    /* we need to build the actual sk_buff from the packet structure */
    pBinding = (PBINDING) Binding;
    lpPacketDescriptor = (LPPACKETDESCRIPTOR) Packet;
    lpMacFragment = (LPFRAGMENTBUFFER) MacHeader;

    /*XXX somebody write a comment about the + 2 on this call... */
    skb = dev_alloc_skb(lpPacketDescriptor->uiPacketSize
                        + lpMacFragment->uiFragmentDataSize + 2);

    if (!skb)
    {
        rc = CNI_W_OUT_OF_DESCRIPTORS;
        goto exit_gracefully;
    }
    /* transfer the packet data into sk_buff */

    switch (pBinding->real_hh_len)
    {
    case ETH_HLEN:
        pMac = skb_put(skb, lpMacFragment->uiFragmentDataSize);
        memcpy(pMac, lpMacFragment->lpFragmentData,
               lpMacFragment->uiFragmentDataSize);
        break;
    case 0:
        pMac = skb->data;
        break;
    case IPPP_MAX_HEADER:
        pMac = skb_put(skb,IPPP_MAX_HEADER);
        break;
    default:
        break;
    };

    pIP = skb_put(skb, lpPacketDescriptor->uiPacketSize);
    CniGetPacketData(Packet, 0, lpPacketDescriptor->uiPacketSize, pIP);

    /* put the mac header on */
    do_gettimeofday(&skb->stamp);

    skb->dev = pBinding->pDevice;

    skb->mac.raw = pMac;
    skb->nh.raw = pIP;

    /*ip header length is in 32bit words */
    skb->h.raw = pIP + (skb->nh.iph->ihl * 4);
    skb->protocol = htons(ETH_P_IP);

    /* send this packet up the NIC driver */
    pBinding->send_stat.rc = pBinding->InjectSend(skb, skb->dev);
    /* if the nic's hard_start_xmit function failed,
       the kernel will queue original packet
       and send to us again.  So, we free the packet that was just built,
       and when the kernel calls dev->hard_start_xmit, we'll start all
       over again... see sch_generic.c:qdisc_restart() for details.
     */
    if (pBinding->send_stat.rc != 0)
    {
        dev_kfree_skb(skb);
    }
    pBinding->send_stat.called = TRUE;

  exit_gracefully:
    /* we have to call Sendcomplete here or else the sendcontext will
     * not be free */
    CNICallbackTable.SendComplete(pBinding, SendContext, Packet);
    return rc;
}
