/**************************************************************************
*           Copyright (c) 2001, Cisco Systems, All Rights Reserved
***************************************************************************
*
*  File:    interceptor.c
*  Date:    04/10/2001
*
***************************************************************************
* This module implements the linux driver.
***************************************************************************/
#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/if_arp.h>
#include <linux/in.h>
#include <linux/ppp_defs.h>
#include <net/ip.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include <net/protocol.h>

#include "linux_os.h"

#include "vpn_ioctl_linux.h"
#include "Cniapi.h"
#include "linuxcniapi.h"
#include "frag.h"

/*methods of the cipsec network device*/
static int interceptor_init(struct net_device *);
static struct net_device_stats *interceptor_stats(struct net_device *dev);
static int interceptor_ioctl(struct net_device *dev, struct ifreq *ifr,
                             int cmd);
static int interceptor_open(struct net_device *dev);
static int interceptor_stop(struct net_device *dev);
static int interceptor_tx(struct sk_buff *skb, struct net_device *dev);

/*helper functions*/
static BINDING *getbindingbydev(struct net_device *dev);
static void do_cleanup();
static int handle_vpnup();
static int handle_vpndown();
static CNIFRAGMENT build_ppp_fake_mac_frag(struct ethhdr *dummy);
static int inline supported_device(struct net_device *dev);
int inline ippp_dev(struct net_device *dev);

/*packet handler functions*/
static int recv_ip_packet_handler(struct sk_buff *skb,
                                  struct net_device *dev,
                                  struct packet_type *type);
static int replacement_dev_xmit(struct sk_buff *skb, struct net_device *dev);

static int handle_netdev_event(struct notifier_block *self, unsigned long,
                               void *);

struct packet_type_funcs
{
    struct packet_type *pt;
    int (*orig_handler_func) (struct sk_buff *, struct net_device *,
                              struct packet_type *);
};
static struct packet_type_funcs original_ip_handler;
uint8 ppp_fake_lcl_addr[ETH_ALEN] = { 45, 45, 45, 45, 45, 45 };

extern CNI_CHARACTERISTICS CNICallbackTable; /* This stores the Plugin's function pointers */
extern PCHAR pcDeviceName;      /* Ignore. Only our pluggin so we don't care about it */

static int vpn_is_up = FALSE;

static int tx_packetcount = 0;
static int rx_packetcount = 0;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)
#define interceptor_name LINUX_VPN_IFNAME
#else
static char interceptor_name[] = LINUX_VPN_IFNAME;
#endif

BINDING Bindings[MAX_INTERFACES];

static struct net_device interceptor_dev = {
    .name = interceptor_name,
    .init = interceptor_init
};
static struct notifier_block interceptor_notifier = {
    .notifier_call = handle_netdev_event,
};

static int __init
interceptor_init(struct net_device *dev)
{
    dev->open = interceptor_open;
    dev->stop = interceptor_stop;
    dev->hard_start_xmit = interceptor_tx;
    dev->get_stats = interceptor_stats;
    dev->do_ioctl = interceptor_ioctl;


    dev->type = ARPHRD_ETHER;
    dev->hard_header_len = ETH_HLEN;
    dev->mtu = 1400;            /* eth_mtu */
    dev->addr_len = ETH_ALEN;
    dev->tx_queue_len = 100;    /* Ethernet wants good queues */

    dev->flags = IFF_BROADCAST | IFF_MULTICAST;
    memset(dev->broadcast, 0xFF, ETH_ALEN);

    return 0;
}

static struct net_device_stats *
interceptor_stats(struct net_device *dev)
{
    static struct net_device_stats es;


    es.rx_packets = rx_packetcount;
    es.rx_errors = 0;
    es.rx_dropped = 0;
    es.rx_fifo_errors = 0;
    es.rx_length_errors = 0;
    es.rx_over_errors = 0;
    es.rx_crc_errors = 0;
    es.rx_frame_errors = 0;
    es.tx_packets = tx_packetcount;
    es.tx_errors = 0;
    es.tx_dropped = 0;
    es.tx_fifo_errors = 0;
    es.collisions = 0;
    es.tx_carrier_errors = 0;
    es.tx_aborted_errors = 0;
    es.tx_window_errors = 0;
    es.tx_heartbeat_errors = 0;

    return (&es);
}

static int
interceptor_ioctl(struct net_device *dev, struct ifreq *ifr, int cmd)
{
    int error = 0;
    static struct ifvpncmd command;

#ifdef MOD_INC_AND_DEC
    MOD_INC_USE_COUNT;
#endif
    switch (cmd)
    {
    case SIOCGVPNCMD:
        {
            uint32 ret_size;

            char *from = (char *) ifr->ifr_ifru.ifru_data;
            error = copy_from_user(&command, from, sizeof(struct ifvpncmd));
            if (error)
            {
                break;
            }

            error = CniPluginIOCTL(command.cmd, (void *) command.data,
                                   command.datalen, sizeof(command.data),
                                   &ret_size);

            if (!error)
            {
                command.datalen = ret_size;
            }
            else
            {
                //XXX set error to a real error value!!!! -E?????
                command.datalen = 0;
            }

            error = copy_to_user(from, &command, sizeof(struct ifvpncmd));
        }
        break;

    case SIOCGVPNIFUP:
        error = handle_vpnup();
        break;

    case SIOCGVPNIFDOWN:
        error = handle_vpndown();
        break;

    default:
        error = -EOPNOTSUPP;
        break;
    }

#ifdef MOD_INC_AND_DEC
    MOD_DEC_USE_COUNT;
#endif
    return error;
}

static int
interceptor_open(struct net_device *dev)
{
    /*stub. our network interface shouldn' ever come up */
    return 1;
}

static int
interceptor_stop(struct net_device *dev)
{
    /*stub. our network interface shouldn' ever come up */
    return 1;
}

static int
interceptor_tx(struct sk_buff *skb, struct net_device *dev)
{
    dev_kfree_skb(skb);
    return 0;
}
static int
add_netdev(struct net_device *dev)
{
    int rc = -1;
    int i = 0;

    if (!supported_device(dev))
    {
        goto exit_gracefully;
    }

    for (i = 0; i < MAX_INTERFACES; i++)
    {
        if (Bindings[i].pDevice == NULL)
        {
            break;
        }
    }
    if (i >= MAX_INTERFACES)
    {
        printk(KERN_DEBUG "%s:exceeded max network devices (%d) at dev %s (%d)",
               __FUNCTION__, MAX_INTERFACES, dev->name, dev->ifindex);
        rc = -1;
        goto exit_gracefully;
    }

    Bindings[i].pDevice = dev;
    /* store the original mtu for this device. */
    Bindings[i].original_mtu = dev->mtu;

    /*replace the original send function with our send function */
    Bindings[i].InjectSend = dev->hard_start_xmit;
    dev->hard_start_xmit = replacement_dev_xmit;

    /*copy in the ip packet handler function and packet type struct */
    Bindings[i].InjectReceive = original_ip_handler.orig_handler_func;
    Bindings[i].pPT = original_ip_handler.pt;

    rc = 0; 

exit_gracefully:
    return rc;
}
static int
remove_netdev(struct net_device *dev)
{
    int rc = -1;
    BINDING *b;

    b = getbindingbydev(dev);

    if (b)
    {   
        rc = 0;
        dev->hard_start_xmit = b->InjectSend;
        memset(b, 0, sizeof(BINDING));
    }
    else
    {
        printk(KERN_DEBUG "%s: missing dev %s (%d)", __FUNCTION__,
               dev->name, dev->ifindex);
    }
    return rc;
}
static int
handle_vpnup(void)
{
    /*temporary structure used to retrieve the registered ip packet handler.
     *it is static because it gets inserted temporarily into a kernel hash
     *table and if things went incredibly wrong it could end up staying there
     */
    static struct packet_type dummy_pt;

    struct net_device *dp = NULL;
    struct packet_type *default_pt = NULL;
    int error = VPNIFUP_SUCCESS, num_target_devices;

    cleanup_frag_queue();

#ifdef MOD_INC_AND_DEC
    MOD_INC_USE_COUNT;
#else
    if (!try_module_get(THIS_MODULE))
    {
        return -EBUSY;
    }
#endif
    if (vpn_is_up)
    {
        error = VPNIFUP_FAILURE;
        return error;
    }

    /* find the handler for inbound IP packets by adding a dummy handler
     * for that packet type into the kernel. Because the packet handlers
     * are stored in a hash table, we'll be able to pull the original 
     * ip packet handler out of the list that dummy_pt was inserted into.*/
    memset(&dummy_pt, 0, sizeof(dummy_pt));
    dummy_pt.type = htons(ETH_P_IP);
    dummy_pt.func = recv_ip_packet_handler;

    dev_add_pack(&dummy_pt);
    /* this should be the original IP packet handler */
    default_pt = PACKET_TYPE_NEXT(&dummy_pt);
    /* there may be more than one other packet handler in our bucket,
     * so look through all the buckets */
    while (default_pt != NULL && default_pt->type != htons(ETH_P_IP))
    {
        default_pt = PACKET_TYPE_NEXT(default_pt);
    }
    if (!default_pt)
    {
        printk(KERN_DEBUG "No default handler found for %x protocol!!\n",
               dummy_pt.type);
        dev_remove_pack(&dummy_pt);
        error = VPNIFUP_FAILURE;
        goto error_exit;
    }
    /*remove the dummy handler handler */
    original_ip_handler.pt = default_pt;
    dev_remove_pack(&dummy_pt);

    /*and replace the original handler function with our function */
    original_ip_handler.orig_handler_func = original_ip_handler.pt->func;
    original_ip_handler.pt->func = recv_ip_packet_handler;

    /* identify the active network devices */
    memset(&Bindings, 0, sizeof(Bindings));

    dp = NULL;
    num_target_devices = 0;
    for (dp = dev_base; dp != NULL; dp = dp->next)
    {
        if (add_netdev(dp) == 0)
        {
            num_target_devices++;
        }
    }

    if (num_target_devices == 0)
    {
        printk(KERN_DEBUG "No network devices detected.\n");
        error = VPNIFUP_FAILURE;
        goto error_exit;
    }
    error = register_netdevice_notifier(&interceptor_notifier);
    if (error)
    {
        goto error_exit;
    }

    vpn_is_up = TRUE;
    return error;

  error_exit:
    do_cleanup();
    vpn_is_up = FALSE;
#ifdef MOD_INC_AND_DEC
    MOD_DEC_USE_COUNT;
#else
    module_put(THIS_MODULE);
#endif
    return error;
}
static void
do_cleanup(void)
{
    int i;

    unregister_netdevice_notifier(&interceptor_notifier);

    cleanup_frag_queue();
    /*restore IP packet handler */
    if (original_ip_handler.pt != NULL)
    {
        original_ip_handler.pt->func = original_ip_handler.orig_handler_func;
    }
    memset(&original_ip_handler, 0, sizeof(original_ip_handler));

    /*restore network devices */
    for (i = 0; i < MAX_INTERFACES; i++)
    {
        struct net_device *dev = Bindings[i].pDevice;
        if (dev)
        {
            remove_netdev(dev);
        }
    }
    memset(&Bindings, 0, sizeof(Bindings));
}
static int
handle_vpndown(void)
{
    int error = VPNIFDOWN_SUCCESS;

    if (!vpn_is_up)
    {
        error = VPNIFDOWN_FAILURE;
        goto exit_gracefully;
    }
    do_cleanup();

    vpn_is_up = FALSE;

#ifdef MOD_INC_AND_DEC
    MOD_DEC_USE_COUNT;
#else
    module_put(THIS_MODULE);
#endif
  exit_gracefully:
    return error;
}
static int
handle_netdev_event(struct notifier_block *self, unsigned long event, void *val)
{
    struct net_device *dev = (struct net_device *) val;

    switch (event)
    {
    case NETDEV_REGISTER:
        add_netdev(dev);
        break;
    case NETDEV_UNREGISTER:
        remove_netdev(dev);
        break;
    default:
        break;
    }
    return 0;
}

static void
reset_inject_status(inject_status * s)
{
    s->called = FALSE;
    s->rc = 0;
}

int inline
ippp_dev(struct net_device *dev)
{
/* This func does not check for syncPPP
   Check for syncPPP must be done in addition to the checks in these
   functions
*/

    int rc = 0;

    if (!dev->name)
    {
        goto exit_gracefully;
    }

    if (strncmp(dev->name, "ippp", 4))
    {
        goto exit_gracefully;
    }

    if (dev->name[4] < '0' || dev->name[4] > '9')
    {
        goto exit_gracefully;
    }

    rc = 1;

  exit_gracefully:
    return rc;
}

static int inline
supported_device(struct net_device *dev)
{
    int rc = 0;

    if (!dev->name)
    {
        goto exit_gracefully;
    }

    if (!strncmp(dev->name, "eth", 3)
        && (dev->name[3] >= '0' && dev->name[3] <= '9'))
    {
        rc = 1;
    }
    else if (!strncmp(dev->name, "wlan", 4)
             && (dev->name[4] >= '0' && dev->name[4] <= '9'))
    {
        rc = 1;
    }
    else if (!strncmp(dev->name, "ppp", 3) && (dev->name[3] >= '0' &&
                                               dev->name[3] <= '9'))
    {
        rc = 1;
    }
    else if (ippp_dev(dev))
    {
        rc = 1;
    }

  exit_gracefully:
    return rc;
}

static BINDING *
getbindingbydev(struct net_device *dev)
{
    int i;

    for (i=0; i <= MAX_INTERFACES; i++)
    {
        BINDING *b = &Bindings[i];
        if (b->pDevice && (dev->ifindex == b->pDevice->ifindex))
        {
            return b;
        }
    }
    return NULL;
}

static CNIFRAGMENT
build_ppp_fake_mac_frag(struct ethhdr *dummy)
{
    CNIFRAGMENT MacHdr = NULL;

    memset(dummy->h_dest, 45, ETH_ALEN);
    memset(dummy->h_source, 45, ETH_ALEN);
    dummy->h_proto = htons(ETH_P_IP);

    CniNewFragment(ETH_HLEN, (char *) dummy, &MacHdr, CNI_USE_BUFFER);
    return MacHdr;
}

static int
recv_ip_packet_handler(struct sk_buff *skb,
                       struct net_device *dev, struct packet_type *type)
{
    int rc2 = 0;
    int tmp_rc = 0;
    CNISTATUS rc = 0;
    CNIPACKET NewPacket = NULL;
    CNIFRAGMENT Fragment = NULL;
    CNIFRAGMENT MacHdr = NULL;
    PVOID lpReceiveContext = NULL;
    ULONG ulFinalPacketSize;
    BINDING *pBinding = NULL;
    struct ethhdr ppp_dummy_buf;
    int hard_header_len;

#ifdef MOD_INC_AND_DEC
    MOD_INC_USE_COUNT;
#endif
    if (strcmp("lo", dev->name) == 0)
    {
        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
        goto exit_gracefully;
    }

    /* Don't handle non-eth non-ppp packets */
    if (!supported_device(dev))
    {
        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
        goto exit_gracefully;
    }

    pBinding = getbindingbydev(dev);

    /* if we don't have a binding, this is a new device that
     *  has been brought up while the tunnel is up. For now,
     *  just pass the packet
     */
    if (!pBinding)
    {
        static int firsttime = 1;
        if (firsttime)
        {
            printk(KERN_DEBUG "RECV: new dev %s detected\n", dev->name);
            firsttime = 0;
        }
        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
        goto exit_gracefully;
    }

    //only need to handle IP packets.
    if (skb->protocol != htons(ETH_P_IP))
    {
        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);
        goto exit_gracefully;
    }

    reset_inject_status(&pBinding->recv_stat);
    if (skb->mac.raw)
    {
        hard_header_len = skb->data - skb->mac.raw;
        if ((hard_header_len < 0) || (hard_header_len > skb_headroom(skb)))
        {
            printk(KERN_DEBUG "bad hh len %d\n", hard_header_len);
            hard_header_len = 0;
        }
    }
    else
    {
        hard_header_len = 0;
    }

    pBinding->real_hh_len = hard_header_len;

    switch (hard_header_len)
    {
    case ETH_HLEN:
        CniNewFragment(ETH_HLEN, skb->mac.raw, &MacHdr, CNI_USE_BUFFER);
        break;
    case IPPP_MAX_HEADER:
    case 0:
        MacHdr = build_ppp_fake_mac_frag(&ppp_dummy_buf);
        break;
    default:
        printk(KERN_DEBUG "unknown mac header length (%d)\n", hard_header_len);
        dev_kfree_skb(skb);
        skb = NULL;
        goto exit_gracefully;
    }

    CniNewFragment(skb->len, skb->data, &Fragment, CNI_USE_BUFFER);
    ulFinalPacketSize = skb->len;

    rc = CNICallbackTable.Receive((void *) &pBinding,
                                  &lpReceiveContext,
                                  &MacHdr,
                                  &Fragment, &NewPacket, &ulFinalPacketSize);

    switch (rc)
    {
    case CNI_CONSUME:
        tmp_rc = CNICallbackTable.ReceiveComplete(pBinding,
                                                  lpReceiveContext, NewPacket);
        dev_kfree_skb(skb);

        if (pBinding->recv_stat.called)
        {
            rc2 = pBinding->recv_stat.rc;
            rx_packetcount++;
        }
        break;
    case CNI_CHAIN:
        tmp_rc = CNICallbackTable.ReceiveComplete(pBinding,
                                                  lpReceiveContext, NewPacket);

        rc2 = original_ip_handler.orig_handler_func(skb, dev, type);

        if (pBinding->recv_stat.called)
        {
            rc2 = pBinding->recv_stat.rc;
        }

        rx_packetcount++;
        break;
    case CNI_DISCARD:
        dev_kfree_skb(skb);
        break;
    default:
        printk(KERN_DEBUG "RECV: Unhandled case in %s rc was %x\n",
               __FUNCTION__, (uint) rc);

        dev_kfree_skb(skb);
    }
  exit_gracefully:
    if (MacHdr)
    {
        CniReleaseFragment(MacHdr, CNI_KEEP_BUFFERS);
    }
    if (Fragment)
    {
        CniReleaseFragment(Fragment, CNI_KEEP_BUFFERS);
    }
#ifdef MOD_INC_AND_DEC
    MOD_DEC_USE_COUNT;
#endif
    return rc2;
}

int
do_cni_send(BINDING * pBinding, struct sk_buff *skb, struct net_device *dev)
{
    int rc2 = 0;
    CNISTATUS rc = 0;
    CNIPACKET Packet = NULL;
    CNIFRAGMENT Fragment = NULL;
    CNIFRAGMENT MacHdr = NULL;
    PVOID lpSendContext = NULL;
    struct ethhdr ppp_dummy_buf;
    int hard_header_len = 0;

    reset_inject_status(&pBinding->send_stat);
    hard_header_len = skb->nh.raw - skb->data;
    pBinding->real_hh_len = hard_header_len;
    switch (hard_header_len)
    {
    case ETH_HLEN:
        CniNewFragment(ETH_HLEN, skb->data, &MacHdr, CNI_USE_BUFFER);
        break;
    case IPPP_MAX_HEADER:
    case 0:
        MacHdr = build_ppp_fake_mac_frag(&ppp_dummy_buf);
        /* note: the PPP device says it's hard_header_len is 4,
         * but skb->data points at the IP header*/
        break;
    default:
        printk(KERN_DEBUG "unknown mac header length (%d)\n",
               skb->dev->hard_header_len);
        dev_kfree_skb(skb);
        skb = NULL;
        goto exit_gracefully;
    }
    CniNewPacket(0, &Packet);
    /*skb->data points to the mac header, the fragment should start
     *with the ip header */
    CniNewFragment(skb->len - hard_header_len,
                   skb->data + hard_header_len, &Fragment, CNI_USE_BUFFER);

    CniAddFragToFront(Packet, Fragment);

    rc = CNICallbackTable.Send((void *) &pBinding,
                               &lpSendContext, &MacHdr, &Packet);

    switch (rc)
    {
    case CNI_DISCARD:
        /* packet was tunneled */
        if (pBinding->send_stat.called)
        {
            rc2 = pBinding->send_stat.rc;

            /*if the packet was tunneled, rc2 should
               now contain the return code from the
               call to the nic's hard_start_xmit function.
               if that function failed, the kernel is going
               to queue this skb and send it to us again later,
               so don't free it. */
            if (rc2 == 0)
            {
                dev_kfree_skb(skb);
            }
        }
        /* packet dropped */
        else
        {
            dev_kfree_skb(skb);
        }
        break;
    case CNI_CHAIN:
        rc2 = pBinding->InjectSend(skb, dev);
        tx_packetcount++;
        break;
    default:
        printk(KERN_DEBUG "Unhandled case in %s rc was %x\n", __FUNCTION__,
               (uint) rc);

        dev_kfree_skb(skb);
        rc2 = 0;
    }
  exit_gracefully:
    if (MacHdr)
    {
        CniReleaseFragment(MacHdr, CNI_KEEP_BUFFERS);
    }
    if (Packet)
    {
        CniReleasePacket(Packet, CNI_KEEP_BUFFERS);
    }
    return rc2;
}
static int
replacement_dev_xmit(struct sk_buff *skb, struct net_device *dev)
{
    int rc2 = 0;
    BINDING *pBinding = 0;

#ifdef MOD_INC_AND_DEC
    MOD_INC_USE_COUNT;
#endif
    pBinding = getbindingbydev(dev);
    /* if we don't have a binding, this is a new device that
     *  has been brought up while the tunnel is up. For now,
     *  just drop the packet.
     */
    if (!pBinding)
    {
        static int firsttime = 1;
        if (firsttime)
        {
            printk(KERN_DEBUG "XMIT: new dev %s detected\n", dev->name);
            firsttime = 0;
        }
        dev_kfree_skb(skb);
        goto exit_gracefully;
    }

    //only need to handle IP packets.
    if (skb->protocol != htons(ETH_P_IP))
    {
        rc2 = pBinding->InjectSend(skb, dev);
        goto exit_gracefully;
    }

    if (need_reorder_frag(skb))
    {
        rc2 = handle_fragment(pBinding, skb, dev);
    }
    else
    {
        rc2 = do_cni_send(pBinding, skb, dev);
    }
  exit_gracefully:
#ifdef MOD_INC_AND_DEC
    MOD_DEC_USE_COUNT;
#endif
    return rc2;
}


int
interceptor_mod_init(void)
{
    int status = 0;
    PCNI_CHARACTERISTICS PCNICallbackTable;
    CNISTATUS rc = CNI_SUCCESS;
    rc = CniPluginLoad(&pcDeviceName, &PCNICallbackTable);

    if (CNI_IS_SUCCESS(rc))
    {

        CNICallbackTable = *PCNICallbackTable;
        CniPluginDeviceCreated();

        if ((status = register_netdev(&interceptor_dev)) != 0)
        {
            printk(KERN_INFO "%s: error %d registering device \"%s\".\n",
                   LINUX_VPN_IFNAME, status, interceptor_dev.name);
            CniPluginUnload();

        }
    }
    if (status == 0)
    {
        printk(KERN_INFO "Cisco Systems VPN Client Version "
               BUILDVER_STRING " kernel module loaded\n");
    }
    return (status);
}

void
interceptor_mod_cleanup(void)
{
    cleanup_frag_queue();
    CniPluginUnload();

    unregister_netdev(&interceptor_dev);
    return;
}

module_init(interceptor_mod_init);
module_exit(interceptor_mod_cleanup);
MODULE_LICENSE("Proprietary");
